/*
 * ltxinput.c
 * Extract, from a LaTeX file, a list FILELIST of all files input and included.
 * Write this list to STDOUT as a tree and/or to a batch file in the format
 *   SET NAME = FILELIST
 * See accompanying documentation for uses as a MSDOS LaTeX utility.
 *
 * Usage: ltxinput [options] filename[.tex]
 *  -q         Quiet.
 *  -n         No batch file created.
 *  -r         Reversed order of list in batch file.
 *  -e         Exit when an input file is found not to exist
 *  -o         Only write existent files to the filelist
 *  -v <name>  Use NAME as the environmental variable to set
 *             - default is LTXINPUT
 *
 * Tested on: MSDOS + TopSpeed C (obscure but cheap)
 *            MSDOS + Djgpp
 * 
 * Author: Jim Green  <j.j.green@@shef.ac.uk>
 *
 * Status: You can do whatever you like with this program subject to
 * the following two conditions:
 * 1) Any modified version of this file may only be
 * distributed if it has a different name.
 * 2) These conditions apply to all modified versions.
 * These are just to avoid any confusion.
 *
 * Acknowlegement: This code is a modification of the file flatex.c
 * by Sariel Har-Peled <sariel@@math.tau.ac.il> which can be obtained
 * from the usual CTAN servers in the directory support\flatex. 
 * Thanks for making this code freely available.
 *
 * Version 0.91 11/22/96   
 *
 * History:
 * 0.1   10/28/96  [JJG] First hack of Har-Peled's flatex.c.
 * 0.11  10/29/96  [JJG] All output to files removed. A lot faster.
 *                       Routines given sensible names.
 *                       All Tabs removed for nice Lgrind output.
 * 0.12  11/02/96  [JJG] Routine addToFilelist added.
 *                       fileList now generated.
 *                       -r option works.
 *                       -v option added.
 * 0.13  11/03/96  [JJG] Routine makeBat added.
 *                       -v option works.
 *                       malloc() problems sorted.
 *                       First working version!
 * 0.14  11/07/96  [JJG] -e option added
 *                       some tidying    
 * 0.15  11/10/96  [JJG] -o option added
 *                       %$\alpha$%-release version   
 *                       some STDOUT aesthetics 
 * 0.9   11/20/96  [JJG] %$\beta$%-release
 * 0.91  11/22/96  [JJG] Memory allocation bug in addToFilelist removed.
 *                       fileList bigger than LINE_SIZE now causes 
 *                       graceful exit
 * 
 * To do:
 * handle %%\verb|\includeonly|%% command
 * 
 */

#include  <stdlib.h>
#include  <stdio.h>
#include  <string.h>
#include  <math.h>
#include  <ctype.h>


/* Static constants. */
#define  LINE_SIZE 1000
#define  FALSE     0
#define  TRUE      1
#define  USE_ARGUMENT( X )   ((void)X)

/* Types */
typedef struct{
  char    verbose, bat, reversed, exit, onlyExistent;
  int     cSpecialInputLevel;
  char    szFullName[LINE_SIZE];
  char    environmentVar[LINE_SIZE];
  }
  structFlags;

/* Static prototypes */
static void SearchFile(char         * szInName,
                       int            level,
                       structFlags  * pFlags,
                       char         * fileList
                       );
static void replaceExt(char * str, char * ext );

/* Start of Code */

static void spacesByLevel(int level){
  while (level > 0){
    printf("   ");
    level--;
    }
  }

static void printHelp(void){
  printf( "\nltxinput - list include/inputs in a LaTeX file\n\n" );
  printf( "Usage:\tltxinput [options] filename[.tex]\n\n" );
  printf( "Options\t-q\t\tQuiet, no output to STDOUT.\n" );
  printf( "\t-n\t\tNo batch file created.\n" );
  printf( "\t-r\t\tReverse order of files in batch file.\n" );
  printf( "\t-e\t\tExit if a file does not exist.\n" );
  printf( "\t-o\t\tOnly existent files in the filelist.\n" );
  printf( "\t-v <NAME>\tUse NAME as the environmental variable\n" );
  printf( "\t\t\t-default is LTXINPUT.\n" );
  printf( "\n" );
  }

static void * myMalloc(unsigned int size){
  void * ptr;
  ptr = malloc( size );
  if (ptr == NULL){
    fprintf( stderr, "Not enough memory" );
    exit( -1 );
    }
  return ptr;
  }

static void addToFilelist(char        * fileList,
                          char        * fileName,
                          structFlags * pFlags){
  char * first;
  char * second;
  static int sizeofFileList = 0;

  first = (char *)myMalloc(LINE_SIZE);
  second = (char *)myMalloc(LINE_SIZE);

  if (pFlags->bat){
    sizeofFileList += strlen(fileName)+1;
    if(sizeofFileList >= LINE_SIZE){
      /* if this happens then fileList contains at least LINE_SIZE/13 files! */
      fprintf(stderr, "ERROR: Length of filelist exceeds %i characters",LINE_SIZE);
      exit(-1);
      }    
    if (pFlags->reversed){
      strcpy(first,fileName);
      strcpy(second,fileList);
      }
    else {
      strcpy(second,fileName);
      strcpy(first,fileList);
      }
    strcat(first, " ");
    strcat(first, second);
    strcpy(fileList, first);
    free(first);
    free(second);
    }
  }

static void handleIncludeCommand(char * line,
                                 char * lpszInclude,
                                 int    level,
                                 structFlags  * pFlags,
                                 char * fileList
                                 ){
  char * lpszBrace, * lpszName, * lpszEndBrace;
  char fInput = 0;

  lpszBrace = NULL;
  if (strncmp(lpszInclude, "\\input", 6) == 0){
    lpszBrace = lpszInclude + 6;
    fInput = 1;
    }
  else if (strncmp(lpszInclude, "\\include", 8)== 0){
    lpszBrace = lpszInclude + 8;
    }
  lpszEndBrace = strchr(lpszBrace, '}');
  if (*lpszBrace != '{' || lpszEndBrace == NULL){
    fprintf(stderr, "ERROR: Expected brace not found.\n\n\tline:%s\n", line);
    exit( -1 );
    }
       
  *lpszEndBrace = 0;
  lpszName = (char *)myMalloc(LINE_SIZE);
  strcpy( lpszName, lpszBrace + 1 );
  if  (! fInput){ 
    replaceExt(lpszName, ".tex");
    }
  SearchFile(lpszName, level + 1, pFlags, fileList);
  lpszEndBrace++;
  while (*lpszEndBrace){
    *line++ = *lpszEndBrace++;
    }
  *line = 0;
  free( lpszName );
  }

static char isBefore(char * lpszA, char * lpszB){
  if (lpszB == NULL) return  TRUE;
  if (lpszA == NULL) return  FALSE;
  if ((int)(lpszA -lpszB) < 0){
    return   TRUE;
    }
  return   FALSE;
  }

static FILE * fopenTex(char * file,
                       char * mode){
  FILE * fl;
  fl = fopen( file, mode );
  if (fl != NULL) return fl;
  replaceExt( file, ".tex" );
  fl = fopen( file, mode );
  return  fl;
  }

static char isTexFileExists(char * file){
  FILE     * fl;
  fl = fopenTex( file, "rt" );
  if  ( fl != NULL ) {
    fclose( fl );
    return  1;
    }
  return  0;
  }

static void addTexExt(char * file){
  FILE * fl;
  fl = fopenTex( file, "rt");
  if  (fl != NULL){
    fclose( fl );
    }
  }

static char is_str_prefix(char * str, char * prefix){
  int len;
  if (str == NULL || prefix == NULL) return  0;
  len = strlen( prefix );
  return  (strncmp( str, prefix, len ) == 0);
  }

static void  SearchFile(char * pSzInName,
                        int    level,
                        structFlags  * pFlags,
                        char   * fileList){
  FILE  * flIn;
  char  * str, * lpszInput, * lpszInclude, * line, * lpszRem, *inc;
  char  * lpszLine, * lpszRemark;
  char  * lpszNewCommand;
  char  cont;
  char  repFlag;
  char  szInName[ 100 ];

  strcpy( szInName, pSzInName );

  line = (char *)myMalloc( LINE_SIZE );
  lpszLine = (char *)myMalloc( LINE_SIZE );
  lpszRemark = (char *)myMalloc( LINE_SIZE );
  
  addTexExt(szInName);
  
  flIn = fopenTex(szInName, "rt");
  if (flIn == NULL){
    if (pFlags->verbose){
      printf("\t");
      spacesByLevel(level);
      printf( "%s (cannot open)\n", szInName );
      }
    if (pFlags->exit){
      fprintf(stderr,"Cannot open %s\n",szInName);
      exit(-1);
      }
    if (! pFlags->onlyExistent){
      addToFilelist(fileList, szInName, pFlags);
      }
    }
  else{
    if (pFlags->verbose){
      printf("\t");
      spacesByLevel(level);
      printf("%s\n", szInName);
      }
    addToFilelist(fileList, szInName, pFlags);
    *lpszRemark = 0;
    while (! feof(flIn)){
      str = fgets( line, LINE_SIZE, flIn );
      if (str == NULL) break;
      strcpy( lpszLine, line );
      lpszRem = strchr( line, '%' );
      if (lpszRem != NULL){
        strcpy( lpszRemark, lpszRem );
        *lpszRem = 0;
        }
      do {
        cont = 0;
        lpszInput = strstr(line, "\\input");
        inc = line;
        do {
          repFlag = 0;
          lpszInclude = strstr(inc, "\\include");
          if (is_str_prefix(lpszInclude, "\\includeversion")
              ||  is_str_prefix( lpszInclude,
              "\\includegraphics")){
            repFlag = 1;
            inc = lpszInclude + 1;
            continue;
            }
          if (is_str_prefix(lpszInclude, "\\includeonly")){
            fprintf(stderr, "WARNING:\"\\includeonly\" command ignored\n");
            inc = lpszInclude + 1;
            repFlag = 1;
            continue;
            }
          if (lpszInclude != NULL && isalpha(lpszInclude[8])){
            fprintf(stderr, "\nWarning: include-like(?) command ignored"
                            " at line:\n\t%s", lpszLine );
            inc = lpszInclude + 1;
            repFlag = 1;
            continue;
            }
          } while (repFlag);
        if (isBefore(lpszInput, lpszInclude)){
          lpszInclude = lpszInput;
          }
        if (lpszInclude != NULL){
          lpszNewCommand = strstr( line, "\\newcommand" );
          if (lpszNewCommand == NULL){
            handleIncludeCommand(line, lpszInclude, level, pFlags, fileList);
            cont = 1;
            }
          }
        } while (cont);
      }
    fclose( flIn );
    }
  free( line );
  free( lpszLine );
  free( lpszRemark );
  }


static void replaceExt(char * str, char * ext ){
  int len, ind;

  len = strlen( str );
  ind = len - 1;
  while (ind >= 0 && 
         str[ ind ] != '.' && 
         str[ ind ] != '\\' &&
         str[ ind ] != '/' 
         ){
    ind--;
    }
  if (ind >= 0 && str[ind] == '.' ){
    str[ ind ] = 0;
    }
  strcat( str, ext );
  }

static void makeBat(char        * fileName,
                    char        * fileList,
                    structFlags * pFlags){

  char             * szOutName;
  FILE             * flOut;
  if (pFlags->bat){
    /* this dosnt need to be this big */
    szOutName = (char *)myMalloc( LINE_SIZE );
    strcpy( szOutName, fileName );
    replaceExt( szOutName, ".bat" );
    flOut = fopen( szOutName, "wt" );
    if   ( flOut == NULL ) {
        fprintf( stderr, "Unable to open file: %s", szOutName );
        exit( -1 );
        }
    fprintf( flOut, "@echo off\nrem generated by ltxinput\nSET %s=%s", pFlags->environmentVar , fileList);
    fclose( flOut );
    if (pFlags->verbose){
      printf( "\nList of files written to %s\n", szOutName );
      }
    }
  }



static void FindInputs(char        * fileName,
                       structFlags * pFlags){
  char   * szInName;
  int    inLen;
  char   * fileList;

  fileList = (char *)myMalloc( LINE_SIZE );
  strcpy(fileList, "");
  szInName = (char *)myMalloc( LINE_SIZE );
  strcpy(szInName, fileName);
  inLen = strlen(szInName);
  if (inLen < 4 ||
     (szInName[inLen] != '.' &&
      strcmp(szInName + inLen - 4, ".tex") != 0)
     ) {
    strcat( szInName, ".tex" );
    }
  if (! isTexFileExists(szInName)){
    if (pFlags->exit){
      if (pFlags->verbose){
        printf("Unable to open file %s\n", szInName );
        }
      exit( -1 );
      }
    if (pFlags->onlyExistent){
      if (pFlags->verbose){
        printf("Unable to open %s - no batch file created\n", szInName );
        }
      exit( -1 );
      }
    else{
      makeBat(fileName, szInName, pFlags);
      exit(0);
      }
    }
  if (pFlags->verbose){
    printf("Reading file %s\n", szInName);
    }
  strcpy( pFlags->szFullName, szInName );
  SearchFile(szInName, 0, pFlags, fileList );
  makeBat(fileName, fileList, pFlags);
  }


static char isFlag(char * str,
                   char  ch){
  if (
       str[ 0 ] == '-' &&
       (str[ 1 ] == ch ||
       str[1] == toupper(ch))
       && (str[2] == 0)
       )
      return  TRUE;
    return  FALSE;
}


int main(int argc, char * argv[]){
  int          ind;
  structFlags  sFlags;
  char * environmentVar; 

  /* parse command line */
  if (argc==1){
     printHelp();
     exit(0);
     }
  /* defaults */
  sFlags.verbose = TRUE;
  sFlags.bat = TRUE;
  sFlags.reversed = FALSE;
  sFlags.exit = FALSE;
  sFlags.onlyExistent = FALSE;
  environmentVar = (char *)myMalloc( LINE_SIZE );
  strcpy(environmentVar, "LTXINPUT");
  strcpy(sFlags.environmentVar, environmentVar);
  for (ind = 1; ind < argc; ind++){
    if (isFlag(argv[ind], 'q')){
      sFlags.verbose = FALSE;
      continue;
      }
    if (isFlag(argv[ind], 'n')){
      sFlags.bat = FALSE;
      continue;
      }
    if (isFlag(argv[ind], 'r')){
      sFlags.reversed = TRUE;
      continue;
      }
    if (isFlag(argv[ind], 'e')){
      sFlags.exit = TRUE;
      continue;
      }
    if (isFlag(argv[ind], 'o')){
      sFlags.onlyExistent = TRUE;
      continue;
      }
    if (isFlag(argv[ind], 'v')){
      ind++;
      strcpy(sFlags.environmentVar, argv[ind]);
      continue;
      }
    }
  if (sFlags.verbose){
    printf("This is LTXinput 0.91\n");
    }
  FindInputs(argv[ind-1], &sFlags);
  return   0;
  }

/*  End of File */